/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.maven.packaging;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.inject.Inject;

import org.apache.camel.tooling.model.JBangCommandModel;
import org.apache.camel.tooling.model.JBangCommandModel.JBangCommand;
import org.apache.camel.tooling.model.JsonMapper;
import org.apache.camel.tooling.util.PackageHelper;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectHelper;
import org.codehaus.plexus.build.BuildContext;
import org.mvel2.templates.TemplateRuntime;

/**
 * Generates documentation for camel-jbang commands.
 */
@Mojo(name = "prepare-jbang-commands-doc", defaultPhase = LifecyclePhase.PROCESS_CLASSES, threadSafe = true,
      requiresDependencyResolution = ResolutionScope.COMPILE)
public class PrepareCamelJBangCommandsDocMojo extends AbstractGeneratorMojo {

    // Marker to identify auto-generated files
    private static final String AUTO_GENERATED_MARKER = "// AUTO-GENERATED by camel-package-maven-plugin";

    /**
     * The documentation directory
     */
    @Parameter(defaultValue = "${project.basedir}/../../../docs/user-manual/modules/ROOT/pages")
    protected File docDir;

    /**
     * The metadata file
     */
    @Parameter(defaultValue = "${project.basedir}/src/generated/resources/META-INF/camel-jbang-commands-metadata.json")
    protected File commandsJsonFile;

    @Inject
    public PrepareCamelJBangCommandsDocMojo(MavenProjectHelper projectHelper, BuildContext buildContext) {
        super(projectHelper, buildContext);
    }

    @Override
    public void execute(MavenProject project) throws MojoFailureException, MojoExecutionException {
        docDir = new File(
                project.getBasedir().getParentFile().getParentFile().getParent(), "docs/user-manual/modules/ROOT/pages");
        commandsJsonFile
                = new File(project.getBasedir(), "src/generated/resources/META-INF/camel-jbang-commands-metadata.json");
        super.execute(project);
    }

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        if (!commandsJsonFile.exists()) {
            getLog().debug("Commands metadata file not found: " + commandsJsonFile);
            return;
        }

        try {
            String json = PackageHelper.loadText(commandsJsonFile);
            JBangCommandModel model = JsonMapper.generateJBangCommandModel(json);

            if (model.getCommands().isEmpty()) {
                getLog().info("No commands found in metadata");
                return;
            }

            // Generate index page
            generateIndexPage(model);

            // Generate individual command pages (including subcommands recursively)
            int totalPages = 0;
            for (JBangCommand cmd : model.getCommands()) {
                totalPages += generateCommandPageRecursive(cmd);
            }

            getLog().info("Generated documentation for " + totalPages + " commands");
        } catch (IOException e) {
            throw new MojoExecutionException("Error generating JBang command documentation", e);
        }
    }

    private void generateIndexPage(JBangCommandModel model) throws MojoExecutionException {
        try (InputStream templateStream = getClass().getClassLoader().getResourceAsStream("jbang-commands.mvel")) {
            if (templateStream == null) {
                throw new MojoExecutionException("Template jbang-commands.mvel not found");
            }

            String template = PackageHelper.loadText(templateStream);
            Map<String, Object> ctx = new HashMap<>();
            ctx.put("commands", model.getCommands());

            String content
                    = (String) TemplateRuntime.eval(template, ctx, Collections.singletonMap("util", MvelHelper.INSTANCE));

            Path indexFile = docDir.toPath().resolve("jbang-commands/camel-jbang-commands.adoc");
            Files.createDirectories(indexFile.getParent());
            updateResource(buildContext, indexFile, content);
            getLog().info("Generated: " + indexFile);
        } catch (IOException e) {
            throw new MojoExecutionException("Error generating index page", e);
        }
    }

    private int generateCommandPageRecursive(JBangCommand cmd) throws MojoExecutionException {
        int count = 1;

        // Generate page for this command
        generateCommandPage(cmd);

        // Recursively generate pages for subcommands
        for (JBangCommand sub : cmd.getSubcommands()) {
            count += generateCommandPageRecursive(sub);
        }

        return count;
    }

    private void generateCommandPage(JBangCommand cmd) throws MojoExecutionException {
        Path cmdFile = docDir.toPath().resolve(cmd.getDocFilePath());

        // Check if file already exists and is not auto-generated (i.e., manually written)
        if (Files.exists(cmdFile)) {
            try {
                String existingContent = Files.readString(cmdFile);
                if (!existingContent.contains(AUTO_GENERATED_MARKER)) {
                    // Skip - this is a manually maintained file
                    getLog().debug("Skipping existing manual file: " + cmdFile);
                    return;
                }
            } catch (IOException e) {
                getLog().warn("Could not read existing file: " + cmdFile);
            }
        }

        try (InputStream templateStream = getClass().getClassLoader().getResourceAsStream("jbang-command-page.mvel")) {
            if (templateStream == null) {
                throw new MojoExecutionException("Template jbang-command-page.mvel not found");
            }

            String template = PackageHelper.loadText(templateStream);
            Map<String, Object> ctx = new HashMap<>();
            ctx.put("command", cmd);

            // Check if an examples file exists for this command
            String examplesFileName = getExamplesFileName(cmd);
            Path examplesFile = docDir.toPath().getParent().resolve("partials/jbang-commands/examples/" + examplesFileName);
            ctx.put("hasExamplesFile", Files.exists(examplesFile));
            ctx.put("examplesFileName", examplesFileName);

            String content
                    = (String) TemplateRuntime.eval(template, ctx, Collections.singletonMap("util", MvelHelper.INSTANCE));

            Files.createDirectories(cmdFile.getParent());
            updateResource(buildContext, cmdFile, content);
            getLog().debug("Generated: " + cmdFile);
        } catch (IOException e) {
            throw new MojoExecutionException("Error generating command page for " + cmd.getName(), e);
        }
    }

    /**
     * Gets the examples file name for a command. For example, "run" -> "run.adoc", "catalog component" ->
     * "catalog-component.adoc"
     */
    private String getExamplesFileName(JBangCommand cmd) {
        String fullName = cmd.getFullName() != null ? cmd.getFullName() : cmd.getName();
        return fullName.replace(" ", "-") + ".adoc";
    }
}
