/*
 * VirtualHostHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.4
 * Created by suhler on 99/06/28
 * Last modified by suhler on 04/11/03 08:30:07
 */

package sunlabs.brazil.handler;

import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import java.io.IOException;

/**
 * Handler for managing virtual hosts using the same server configuration.
 * This prefixes the host name (from the http "host" header, with the
 * port prtion removed) onto the
 * url and passes the request along.
 * If no host is provided, the host "default" is used instead.
 * <br>
 * If hosts require their own server configurations,
 * use the {@link MultiHostHandler} instead.
 * <p>
 * Configuration parameters:
 * <dl class=props>
 * <dt> maproot
 * <dd> If set upon server startup, this handler changes the "root" property
 *      instead of the "url" property, by appending the "host" onto the
 *	document root, instead of prepending the "host" to the url.
 * <dt> [prefix].[host].
 * <dd> If the "mapping "property exists that matches the incoming
 *      "host" name, then
 *	that value is used instead of [host] to rewrite the "url" or "root".
 * <dt> default
 * <dd> If set, then all hosts for which no mappings are defined are mapped
 *      to the value of this property.
 * </dl>
 * With no configuration options, each virtual host document root is in
 * a subdirectory whose name matches the host (e.g. www.foo.com).  The
 * "maproot" property changes the mechanism how virtual roots are
 * distingished: by URL or by document root.  The "mapping" properties are 
 * used to choose a name for the subdirectory that differs from the
 * virtual hostname.  Finally, if "default" is set, then virtual hosts
 * with no subdirectory are all shunted into the subdirectory specified.
 * 
 * @author		Stephen Uhler
 * @version		2.1, 02/10/01
 */

public class VirtualHostHandler implements Handler {
    boolean mapRoot = false;	// map the root, don't rewrite the url
    String prefix;

    public boolean
    init(Server server, String prefix) {
        this.prefix = prefix;
	if (server.props.getProperty(prefix + "maproot") != null) {
	    mapRoot=true;
	}
	server.log(Server.LOG_DIAGNOSTIC, prefix,
	    "Starting virtual host handler (maproot=" + mapRoot + ")");
	return true;
    }

    /**
     * Either look for host header, tack on front of url,
     * or modify the "root" property
     */

    public boolean
    respond(Request request) throws IOException {
    	String host = (String) request.headers.get("Host");
    	if (host == null) {
    	    host = "default";
	} else {
            host = host.toLowerCase();
	}
	int index = host.indexOf(":");
	if (index >= 0) {
	    host = host.substring(0,host.indexOf(":"));
	}
	String map = request.props.getProperty(prefix + host);
	if (map != null) {
	    request.log(Server.LOG_DIAGNOSTIC, "mapping: " + host + " to " +map);
	    host = map;
	} else if ((map=request.props.getProperty(prefix + "default")) != null) {
	    request.log(Server.LOG_DIAGNOSTIC, "default: " + host + " to " +map);
            host = map;
	}
	String root = request.props.getProperty("root",".") + "/" + host;
	if (mapRoot) {
	    request.props.put("root", root);
	    request.log(Server.LOG_DIAGNOSTIC, "root changed to: " + root);
	} else {
	    request.log(Server.LOG_DIAGNOSTIC, host +
		    " added to url " + request.url);
	    request.url = "/" + host.toLowerCase() + request.url;
	}
	return false;
    }
}
