/*
 * RolesHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1998-2003 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.2
 * Created by suhler on 98/09/30
 * Last modified by suhler on 03/07/15 09:11:13
 */

package sunlabs.brazil.handler;

import java.io.FileInputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;
import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;

/**
 * Handler for associating roles with an id.  This is a placeholder
 * until the SunEconomy gets integrated in.  It looks for an "id" in the 
 * request, looks it up in a property file, then adds the value of the
 * id into the request.  It may be used in conjunction with
 * {@link AclSwitchHandler} to provide role based web access.
 * Properties:
 * <dl class=props>
 * <dt>prefix, suffix, glob, match
 * <dd>Specify the URL that triggers this handler.
 * (See {@link MatchString}).
 * <dt>SessionID
 * <dd>The property to use to look up the id.  Defaults to "SessionID".
 * <dt>roleName
 * <dd>The property to place the result of the id lookup into. 
 *     Defaults to "roleName";
 * <dt>mapFile
 * <dd>The absolute path to the java properties file containing the it to role mapping.
 * </dl>
 *
 * @author		Stephen Uhler
 * @version		 @(#) RolesHandler.java 2.2 03/07/15 09:11:13
 */

public class RolesHandler implements Handler {
    Properties map;		// The authorization mapping table
    String propsPrefix;		// My prefix in the global properties file
    String urlPrefix;		// The prefix to look for to map
    String idKey;	// The property name for the token id
    String roleKey;	// The property name for the roles
    boolean check = true;	// only put results in request, don't return

    /**
     * Handler configuration property <b>SessionID</b>.
     * The request property name to find the id string.
     * Defaults to id.
     */

    public static final String ID_KEY = "SessionID";	// property key for token id

    /**
     * Handler configuration property <b>roleName</b>.
     * The request property name to place the roles into.
     * Defaults to roles.
     */

    public static final String ROLE_KEY = "roleName";	// property key for token roles

    /**
     * Handler configuration property <b>mapFile</b>.
     * The path to the java properties file containing the id
     * to roles mapping.  The roles are a whitespace delimited list
     * of ascii role names.
     */

    public static final String MAP = "mapFile";	// properties file

    /**
     * Handler configuration property <b>check</b>.
     * If true, the results are out into the request object.
     * Otherwise, they are returned in a text/plain java properties
     * formatted document, which can be used with the
     * @see RemoteStsHandler.
     */

    static final String CHECK = "check"; // how results are returned
    MatchString isMine;            // check for matching url

    public boolean
    init(Server server, String prefix) {
	propsPrefix = prefix;
	isMine = new MatchString(prefix, server.props);
	String mapFile = server.props.getProperty(propsPrefix + MAP,"");
	map = new Properties();
	try {
	    FileInputStream in = new FileInputStream(mapFile);
	    map.load(in);
	    in.close();
	} catch (Exception e) {
	    server.log(Server.LOG_ERROR, prefix, propsPrefix + MAP + ": ("
			+ mapFile + ") " + e.toString());
	    return false;
	}
	idKey = server.props.getProperty(propsPrefix + ID_KEY, "id");
	roleKey = server.props.getProperty(propsPrefix + ROLE_KEY, "roles");
	check = server.props.getProperty(
		propsPrefix + CHECK, "y").startsWith("y");
	return true;
    }

    /** 
     * Dispatch and handle the request.
     * This version looks at the request for the id, looks it up in the
     * table, and adds the value, if available
     */

    public boolean
    respond(Request request) throws IOException {
	if (!isMine.match(request.url)) {
	    return false;
        }
	String id = (String) request.props.get(idKey);
	// System.out.println("ids: " + map);
	// System.out.println("props: " + request.props);
	if (id != null && map.containsKey(id)) {
	    String value = (String) map.get(id);
	    request.log(Server.LOG_DIAGNOSTIC, "Mapping: " + id + "->" + value);
	    request.props.put(roleKey,value);
	}
	if (!check) {
	    StringBuffer result = new StringBuffer("");
	    Enumeration keys = request.props.keys();
	    while(keys.hasMoreElements()) {
		String key = (String) keys.nextElement();
		result.append(key + "=" + request.props.get(key) + "\n");
	    }
	    request.sendResponse(result.toString(),"text/plain");
	    return true;
	} else {
	    return false;
	}
    }
}
