/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import * as React from "react";
import { useCallback, useLayoutEffect, useMemo, useState } from "react";
import { Spinner } from "@patternfly/react-core/dist/js/components/Spinner";
import { Title } from "@patternfly/react-core/dist/js/components/Title";
import { Bullseye } from "@patternfly/react-core/dist/js/layouts/Bullseye";
import { useEditorEnvelopeI18nContext } from "../i18n";

export function LoadingScreen(props: { loading: boolean; styleTag?: string }) {
  const [mustRender, setMustRender] = useState(true);
  const { i18n } = useEditorEnvelopeI18nContext();

  const onAnimationEnd = useCallback((e: React.AnimationEvent<HTMLDivElement>) => {
    e.preventDefault();
    e.stopPropagation();
    setMustRender(false);
  }, []);

  const loadingScreenClassName = useMemo(() => {
    if (props.loading) {
      return "";
    }
    return "loading-finished";
  }, [props.loading]);

  useLayoutEffect(() => {
    if (props.loading) {
      setMustRender(true);
    }
  }, [props.loading]);

  const style = (props.styleTag ? `${props.styleTag} ` : "") + "kie-tools--loading-screen";

  return (
    (mustRender && (
      <div id="loading-screen" className={style}>
        <div
          className={`kie-tools--loading-screen ${loadingScreenClassName}`}
          onAnimationEnd={onAnimationEnd}
          data-testid={"loading-screen-div"}
        >
          <Bullseye>
            <div className={"kie-tools--loading-screen-spinner"}>
              <div>
                <Spinner />
              </div>
              <Title headingLevel={"h5"}>{i18n.loadingScreen.loading}</Title>
            </div>
          </Bullseye>
        </div>
      </div>
    )) || <></>
  );
}
